<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;

class AsgApi extends Controller
{
    protected $baseUrl;
    protected $userAgent;
    protected $signature;

    public function __construct()
    {
        $api = DB::table('api_providers')->find(1);

        $this->baseUrl = $api->url;
        $this->userAgent = $api->apikey;
        $this->signature = $api->secretkey;
    }

    protected function basePayload(array $extra = []): array
    {
        return array_merge([
            'user_agent' => $this->userAgent,
            'signature' => $this->signature,
        ], $extra);
    }

    protected function sendRequest(array $payload, string $endpoint)
    {
        $response = Http::asForm()->withHeaders([
            'User-Agent' => 'Mozilla/5.0'
        ])->post($endpoint, $payload);

        if ($response->failed()) {
            throw new \Exception('API request failed: ' . $response->body());
        }

        return $response->json();
    }

    public function infoAgent()
    {
        return $this->sendRequest($this->basePayload(), "{$this->baseUrl}?cmd=info_agent");
    }

    public function infoProvider()
    {
        return $this->sendRequest($this->basePayload(), "{$this->baseUrl}?cmd=info_list_provider");
    }

    public function infoGame($provider)
    {
        return $this->sendRequest($this->basePayload(['provider' => $provider]), "{$this->baseUrl}?cmd=info_list_game");
    }

    public function infoMember($userMember)
    {
        return $this->sendRequest($this->basePayload(['user_member' => $userMember]), "{$this->baseUrl}?cmd=info_member");
    }

    public function createMember($userMember)
    {
        return $this->sendRequest($this->basePayload(['user_member' => $userMember]), "{$this->baseUrl}?cmd=create_member");
    }

    public function depositMember($userMember, $deposit)
    {
        return $this->sendRequest($this->basePayload([
            'user_member' => $userMember,
            'amount' => $deposit,
        ]), "{$this->baseUrl}?cmd=deposit_member");
    }

    public function withdrawMember($userMember, $withdraw)
    {
        return $this->sendRequest($this->basePayload([
            'user_member' => $userMember,
            'amount' => $withdraw,
        ]), "{$this->baseUrl}?cmd=withdraw_member");
    }

    public function launchGame($userMember, $gameCode)
    {
        return $this->sendRequest($this->basePayload([
            'user_member' => $userMember,
            'game_code' => $gameCode,
        ]), "{$this->baseUrl}?cmd=launch_game");
    }

    public function infoBetHistory($dateStart, $dateEnd)
    {
        return $this->sendRequest($this->basePayload([
            'date_start' => $dateStart,
            'date_end' => $dateEnd,
        ]), "{$this->baseUrl}?cmd=info_bet_history");
    }

    public function infoBetMember($userMember, $dateStart, $dateEnd)
    {
        return $this->sendRequest($this->basePayload([
            'user_member' => $userMember,
            'date_start' => $dateStart,
            'date_end' => $dateEnd,
        ]), "{$this->baseUrl}?cmd=info_bet_member");
    }
}
